/*
 * Decompiled with CFR 0.152.
 */
package com.neckarlabs.imgstopdf;

import com.neckarlabs.imgstopdf.exception.NonRecoverableException;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.List;
import javax.imageio.ImageIO;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.apache.pdfbox.pdmodel.PDPageContentStream;
import org.apache.pdfbox.pdmodel.common.PDRectangle;
import org.apache.pdfbox.pdmodel.graphics.image.LosslessFactory;
import org.apache.pdfbox.pdmodel.graphics.image.PDImageXObject;
import org.imgscalr.Scalr;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ImgsToPdf {
    private static final Logger log = LoggerFactory.getLogger(ImgsToPdf.class);
    private static final String DEFAULT_PDF_NAME = "imgs-in-pdf.pdf";
    private int rows;
    private int cols;
    private float pageMargin = 20.0f;
    private float spacing = 10.0f;
    private float resizeFactor = 2.5f;
    private float maxImgWidth;
    private float maxImgHeight;

    public static void printUsage() {
        log.info("\n\nUSAGE: java -jar imgstopdf-1.0.0.jar \"<argument-1>\" \"<argument-2>\"\n<argument-1>: Path to image folder. Supported image types are JPG and PNG\n<argument-2>: Optional grid size of form rows x cols, e.g. 4 x 3. If omitted default 3 x 2 is used. ");
    }

    public static void main(String[] args) {
        try {
            ImgsToPdf imgsToPdf;
            if (args.length < 1 || args.length > 2) {
                ImgsToPdf.printUsage();
                return;
            }
            if (args.length == 2) {
                String[] nums = args[1].split("x");
                try {
                    int rows = Integer.valueOf(nums[0].trim());
                    int cols = Integer.valueOf(nums[1].trim());
                    imgsToPdf = new ImgsToPdf(rows, cols);
                }
                catch (NumberFormatException e) {
                    log.error("Invalid grid size");
                    return;
                }
            } else {
                imgsToPdf = new ImgsToPdf();
            }
            imgsToPdf.initMaxImageSize();
            List<String> imgsPaths = imgsToPdf.readImgs(args[0]);
            String pdfName = args[0] + "/" + DEFAULT_PDF_NAME;
            imgsToPdf.createPdf(pdfName, imgsPaths);
        }
        catch (NonRecoverableException e) {
            log.error(e.getMessage(), e.getCause());
        }
        catch (Exception e) {
            log.error("An error occured", e);
        }
    }

    private ImgsToPdf() {
        this(3, 2);
    }

    private ImgsToPdf(int rows, int cols) {
        this.rows = rows;
        this.cols = cols;
    }

    private void initMaxImageSize() {
        PDPage pdPage = new PDPage(PDRectangle.A4);
        float width = pdPage.getMediaBox().getWidth();
        float height = pdPage.getMediaBox().getHeight();
        this.maxImgWidth = (width - 2.0f * this.pageMargin - (float)(this.cols - 1) * this.spacing) / (float)this.cols;
        this.maxImgHeight = (height - 2.0f * this.pageMargin - (float)(this.rows - 1) * this.spacing) / (float)this.rows;
    }

    private List<String> readImgs(String path) {
        final ArrayList<String> imgsPaths = new ArrayList<String>();
        SimpleFileVisitor<Path> fileVisitor = new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                String fileName = file.getFileName().toString().toLowerCase();
                if (fileName.toLowerCase().endsWith(".jpg") || fileName.toLowerCase().endsWith(".jpeg") || fileName.toLowerCase().endsWith(".png")) {
                    log.info("Found supported image {}", (Object)file.toString());
                    imgsPaths.add(file.toString());
                }
                return FileVisitResult.CONTINUE;
            }
        };
        try {
            Files.walkFileTree(Paths.get(path, new String[0]), (FileVisitor<? super Path>)fileVisitor);
        }
        catch (IOException e) {
            throw new NonRecoverableException("Failed to read images", e);
        }
        return imgsPaths;
    }

    private void createPdf(String pdfName, List<String> imgsPaths) {
        try (PDDocument pdDocument = new PDDocument();){
            for (int i = 0; i < imgsPaths.size(); ++i) {
                log.info("Processing image {} of {}", (Object)(i + 1), (Object)imgsPaths.size());
                BufferedImage bi = this.resizeImage(imgsPaths.get(i), this.resizeFactor);
                PDImageXObject pdImage = LosslessFactory.createFromImage(pdDocument, bi);
                if (i % (this.rows * this.cols) == 0) {
                    pdDocument.addPage(new PDPage(PDRectangle.A4));
                }
                PDPage pdPage = pdDocument.getPage(i / (this.rows * this.cols));
                try (PDPageContentStream contentStream = new PDPageContentStream(pdDocument, pdPage, PDPageContentStream.AppendMode.APPEND, true);){
                    int col = i % this.cols;
                    int row = i % (this.rows * this.cols) / this.cols;
                    float scaleWidth = this.maxImgWidth / (float)pdImage.getWidth();
                    float scaleHeight = this.maxImgHeight / (float)pdImage.getHeight();
                    float scale = Math.min(scaleWidth < 1.0f ? scaleWidth : 1.0f, scaleHeight < 1.0f ? scaleHeight : 1.0f);
                    float imgWidth = (float)pdImage.getWidth() * scale;
                    float imgHeight = (float)pdImage.getHeight() * scale;
                    float x = this.pageMargin + (this.spacing + this.maxImgWidth) * (float)col + (this.maxImgWidth - imgWidth) / 2.0f;
                    float y = this.pageMargin + (this.spacing + this.maxImgHeight) * (float)(this.rows - 1 - row) + (this.maxImgHeight - imgHeight) / 2.0f;
                    contentStream.drawImage(pdImage, x, y, imgWidth, imgHeight);
                    continue;
                }
            }
            pdDocument.save(pdfName);
        }
        catch (IOException e) {
            throw new NonRecoverableException("Failed to create PDF", e);
        }
    }

    private BufferedImage resizeImage(String path, float factor) {
        try {
            BufferedImage img = ImageIO.read(new File(path));
            int width = img.getWidth();
            int height = img.getHeight();
            if ((float)width > factor * this.maxImgWidth || (float)height > factor * this.maxImgHeight) {
                img = Scalr.resize(img, (int)(factor * this.maxImgWidth), (int)(factor * this.maxImgHeight), Scalr.OP_ANTIALIAS);
            }
            return img;
        }
        catch (IOException e) {
            throw new NonRecoverableException("Error while reading file", e);
        }
    }
}

